Updater = {}

Updater.maxRetries = 5

local updaterWindow
local loadModulesFunction
local scheduledEvent
local httpOperationId = 0

-- Hardcoded Ed25519 public key (base64 of 32 raw bytes).
-- Replace the placeholder with your real public key. Keep it base64-encoded.
local UPDATER_PUBKEY_B64 = 'ZLp/IMMKMxp++9BKXN5JWINb63AXyfw5+MYAQ2ePpSM='

local function canonicalizeManifest(data)
  if type(data) ~= 'table' then return nil end
  if type(data.url) ~= 'string' then return nil end
  if type(data.sha256) ~= 'table' then return nil end
  local lines = { data.url }
  local keys = {}
  for k,_ in pairs(data.sha256) do table.insert(keys, k) end
  table.sort(keys)
  for _,k in ipairs(keys) do
    table.insert(lines, k .. '=' .. data.sha256[k])
  end
  if type(data.binary) == 'table' and type(data.binary.file) == 'string' and type(data.binary.sha256) == 'string' then
    table.insert(lines, 'binary:' .. data.binary.file .. '=' .. data.binary.sha256)
  end
  return table.concat(lines, '\n')
end

local function onLog(level, message, time)
  if level == LogError then
    Updater.error(message)
    g_logger.setOnLog(nil)
  end
end

local function loadModules()
  if loadModulesFunction then
    local tmpLoadFunc = loadModulesFunction
    loadModulesFunction = nil
    tmpLoadFunc()
  end
end

local function downloadFiles(url, files, index, retries, doneCallback)
  if not updaterWindow then return end
  local entry = files[index]
  if not entry then -- finished
    return doneCallback()
  end
  local file = entry[1]
  local file_checksum = entry[2]

  if retries > 0 then
    updaterWindow.downloadStatus:setText(tr("Downloading (%i retry):\n%s", retries, file))
  else
    updaterWindow.downloadStatus:setText(tr("Downloading:\n%s", file))
  end
  updaterWindow.downloadProgress:setPercent(0)
  updaterWindow.mainProgress:setPercent(math.floor(100 * index / #files))

  httpOperationId = HTTP.download(url .. file, file,
    function(file, checksum, err, sha256)
      if not err and (not sha256 or sha256:len() == 0) then
        err = 'Missing SHA-256 from download path for ' .. file
      end
      local expected = file_checksum
      if not err and sha256:lower() ~= expected:lower() then
        err = "Invalid checksum of: " .. file .. ".\nShould be " .. expected .. ", is: " .. sha256
      end
      if err then
        if retries >= Updater.maxRetries then
          Updater.error("Can't download file: " .. file .. ".\nError: " .. err)
        else
          scheduledEvent = scheduleEvent(function()
            downloadFiles(url, files, index, retries + 1, doneCallback)
          end, 250)
        end
        return
      end
      downloadFiles(url, files, index + 1, 0, doneCallback)
    end,
    function(progress, speed)
      updaterWindow.downloadProgress:setPercent(progress)
      updaterWindow.downloadProgress:setText(speed .. " kbps")
    end)
end

local function updateFiles(data, keepCurrentFiles)
  if not updaterWindow then return end

  if type(data) ~= "table" then
    return Updater.error("Invalid data from updater api (not table)")
  end

  if type(data.error) == 'string' and data.error:len() > 0 then
    return Updater.error(data.error)
  end

  if type(data.sha256) ~= 'table' or type(data.url) ~= 'string' or data.url:len() < 4 then
    return Updater.error("Invalid data from updater api: " .. json.encode(data, 2))
  end

  if data.keepFiles then
    keepCurrentFiles = true
  end

  local newFiles = false
  local finalFiles = {}
  -- We require SHA-256; compute local hashes lazily
  local localFiles = {}

  local toUpdate = {}
  local toUpdateFiles = {}
  -- keep all files or files from data/things
  for file, checksum in pairs(localFiles) do
    if keepCurrentFiles or string.find(file, "data/things") then
      table.insert(finalFiles, file)
    end
  end

  -- update files
  local serverMap = data.sha256
  for file, checksum in pairs(serverMap) do
    table.insert(finalFiles, file)
    local need = false
    -- compute local sha256 only when necessary
    if not g_resources.fileExists(file) then
      need = true
    else
      local my = g_resources.fileSha256(file)
      need = (my ~= checksum)
    end
    if need then
      table.insert(toUpdate, { file, checksum })
      table.insert(toUpdateFiles, file)
      newFiles = true
    end
  end

  -- update binary
  local binary = nil
  if type(data.binary) == "table" and data.binary.file and data.binary.file:len() > 1 then
    if type(data.binary.sha256) ~= 'string' or data.binary.sha256:len() == 0 then
      return Updater.error('Updater manifest missing binary.sha256')
    end
    local selfSha = g_resources.selfSha256()
    if selfSha:len() > 0 and selfSha ~= data.binary.sha256 then
      binary = data.binary.file
      table.insert(toUpdate, { binary, data.binary.sha256 })
    end
  end

  if #toUpdate == 0 then -- nothing to update
    updaterWindow.mainProgress:setPercent(100)
    scheduledEvent = scheduleEvent(Updater.abort, 20)
    return
  end

  -- update of some files require full client restart
  local forceRestart = false
  local reloadModules = false
  local forceRestartPattern = { "init.lua", "corelib", "updater", "otmod", "build-version.txt" }
  for _, file in ipairs(toUpdate) do
    for __, pattern in ipairs(forceRestartPattern) do
      if string.find(file[1], pattern) then
        forceRestart = true
      end
      if not string.find(file[1], "data/things") then
        reloadModules = true
      end
    end
  end

  updaterWindow.status:setText(tr("Updating %i files", #toUpdate))
  updaterWindow.mainProgress:setPercent(0)
  updaterWindow.downloadProgress:setPercent(0)
  updaterWindow.downloadProgress:show()
  updaterWindow.downloadStatus:show()
  updaterWindow.changeUrlButton:hide()

  downloadFiles(data["url"], toUpdate, 1, 0, function()
    updaterWindow.status:setText(tr("Updating client (may take few seconds)"))
    updaterWindow.mainProgress:setPercent(100)
    updaterWindow.downloadProgress:hide()
    updaterWindow.downloadStatus:hide()
    scheduledEvent = scheduleEvent(function()
      local restart = binary or (not loadModulesFunction and reloadModules) or forceRestart
      if newFiles then
        g_resources.updateFiles(toUpdateFiles, not restart)
      end

      if binary then
        g_resources.updateExecutable(binary)
      end

      if restart then
        g_app.restart()
      else
        if reloadModules then
          g_textures.clearCache()
          g_modules.reloadModules()
        end
        Updater.abort()
      end
    end, 100)
  end)
end

-- public functions
function Updater.init(loadModulesFunc)
  -- dev mode: skip updater in local environment
  if isLocalEnv and isLocalEnv() then
    loadModulesFunction = loadModulesFunc
    Updater.abort()
    return
  end

  -- macOS: disable updater (server manifest is legacy for now)
  if g_app.getOs and g_app.getOs() == 'mac' then
    loadModulesFunction = loadModulesFunc
    Updater.abort()
    return
  end

  g_logger.setOnLog(onLog)
  loadModulesFunction = loadModulesFunc
  Updater.check()
end

function Updater.terminate()
  loadModulesFunction = nil
  Updater.abort(true)
end

function Updater.abort(terminate)
  HTTP.cancel(httpOperationId)
  removeEvent(scheduledEvent)
  if updaterWindow then
    updaterWindow:destroy()
    updaterWindow = nil
  end
  loadModules()
  if not terminate then
    signalcall(g_app.onUpdateFinished, g_app)
  end
end

function Updater.check(args)
  if updaterWindow then return end

  updaterWindow = g_ui.displayUI('updater')
  updaterWindow:show()
  updaterWindow:focus()
  updaterWindow:raise()

  local updateData = nil
  local function progressUpdater(value)
    removeEvent(scheduledEvent)
    if value == 100 then
      return Updater.error(tr("Timeout"))
    end
    if updateData and (value > 60 or (not g_platform.isMobile() or not ALLOW_CUSTOM_SERVERS or not loadModulesFunc)) then -- gives 3s to set custom updater for mobile version
      return updateFiles(updateData)
    end
    scheduledEvent = scheduleEvent(function() progressUpdater(value + 1) end, 100)
    updaterWindow.mainProgress:setPercent(value)
  end
  progressUpdater(0)

  httpOperationId = HTTP.postJSON(Services.updater, {
    version = g_app.getBuildRevision(),
    build = g_app.getVersion(),
    os = g_app.getOs(),
    platform = g_window.getPlatformType(),
    args = args or {}
  }, function(data, err)
    if err then
      return Updater.error(err)
    end
    -- Enforce signed manifest
    if type(data) ~= 'table' or type(data.sha256) ~= 'table' then
      return Updater.error('Updater manifest missing sha256 map')
    end
    if type(data.sig) ~= 'string' or data.sig == '' then
      return Updater.error('Missing signature in updater manifest')
    end
    local pkB64 = UPDATER_PUBKEY_B64
    if not pkB64 or pkB64 == '' or pkB64:match('^REPLACE_') then
      return Updater.error('Updater public key not configured in build')
    end
    local msg = canonicalizeManifest(data)
    if not msg then
      return Updater.error('Invalid manifest structure for signature verification')
    end
    local ok = g_crypt.ed25519VerifyBase64(msg, data.sig, pkB64)
    if not ok then
      return Updater.error('Manifest signature verification failed')
    end

    updateData = data
  end)
end

function Updater.error(message)
  removeEvent(scheduledEvent)
  if not updaterWindow then return end
  displayErrorBox(tr("Updater Error"), message).onOk = function()
    Updater.abort()
  end
end
